<?php
/*
 * Plugin Name: Bongo Pay BD Payment Gateway
 * Plugin URI: https://bongopaybd.com
 * Description: Accept Bkash, Nagad, Rocket, PayPal, and all Bangladeshi gateways securely via Bongo Pay BD.
 * Author: Bongo Pay BD
 * Author URI: https://bongopaybd.com
 * Version: 2.1.0
 * Requires at least: 5.2
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bongopay-bd
 */

add_action('plugins_loaded', 'bongopay_bd_init_gateway_class');

function bongopay_bd_init_gateway_class()
{
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_BongoPay_BD_Gateway extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'bongopay_bd';
            $this->icon = 'https://bongopaybd.com/public/uploads/admin/356a192b7913b04c54574d18c28d46e6395428ab/1761616910_f122636019993e95195b.png';
            $this->has_fields = false;
            $this->method_title = __('Bongo Pay BD', 'bongopay-bd');
            $this->method_description = __('Pay securely with Bongo Pay BD (Bkash, Nagad, Rocket, PayPal, etc.)', 'bongopay-bd');

            $this->supports = array('products');

            $this->init_form_fields();
            $this->init_settings();

            $this->title       = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled     = $this->get_option('enabled');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_' . strtolower(get_class($this)), array($this, 'handle_webhook'));
        }

        /** Admin Settings */
        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title'       => 'Enable/Disable',
                    'label'       => 'Enable Bongo Pay BD Payment',
                    'type'        => 'checkbox',
                    'default'     => 'no'
                ),
                'mode' => array(
                    'title'       => 'Mode',
                    'type'        => 'select',
                    'description' => 'Choose between Live or Test (Sandbox) mode.',
                    'default'     => 'live',
                    'options'     => array(
                        'live' => 'Live',
                        'test' => 'Test (Sandbox)'
                    ),
                ),
                'title' => array(
                    'title'       => 'Title',
                    'type'        => 'text',
                    'default'     => 'Bongo Pay BD',
                    'desc_tip'    => true,
                ),
                'apikeys' => array(
                    'title'       => 'API Key',
                    'type'        => 'text',
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'currency_rate' => array(
                    'title'       => 'USD Rate (if needed)',
                    'type'        => 'number',
                    'default'     => '110',
                    'desc_tip'    => true,
                ),
                'is_digital' => array(
                    'title'       => 'Digital Product Mode',
                    'label'       => 'Enable for instant delivery',
                    'type'        => 'checkbox',
                    'default'     => 'no'
                ),
                'payment_site' => array(
                    'title'             => 'Base Payment URL',
                    'type'              => 'text',
                    'default'           => 'https://secure.bongopaybd.com/',
                    'desc_tip'          => true,
                    'custom_attributes' => array('readonly' => 'readonly'),
                ),
            );
        }

        /** Process Payment */
        public function process_payment($order_id)
        {
            global $woocommerce;
            $order = wc_get_order($order_id);
            $current_user = wp_get_current_user();

            $total = $order->get_total();
            if ($order->get_currency() == 'USD') {
                $total = $total * $this->get_option('currency_rate');
            }

            if ($order->get_status() != 'completed') {
                $order->update_status('pending', __('Redirecting to Bongo Pay BD...', 'bongopay-bd'));
            }

            $mode = $this->get_option('mode');
            $base_url = ($mode === 'test')
                ? 'https://sandbox.bongopaybd.com/'
                : 'https://secure.bongopaybd.com/';

            $data = array(
                "cus_name"    => sanitize_text_field($current_user->user_firstname),
                "cus_email"   => sanitize_email($current_user->user_email),
                "amount"      => $total,
                "webhook_url" => esc_url(site_url('/?wc-api=wc_bongopay_bd_gateway&order_id=' . $order->get_id())),
                "success_url" => esc_url($this->get_return_url($order)),
                "cancel_url"  => esc_url(wc_get_checkout_url())
            );

            $header = array(
                "api" => sanitize_text_field($this->get_option('apikeys')),
                "url" => $base_url . "api/payment/create"
            );

            $response = $this->create_payment($data, $header);
            $data = json_decode($response, true);

            if (empty($data['payment_url'])) {
                wc_add_notice(__('Bongo Pay BD payment URL not received. Please try again.', 'bongopay-bd'), 'error');
                return;
            }

            return array(
                'result'   => 'success',
                'redirect' => esc_url($data['payment_url'])
            );
        }

        /** Create Payment CURL */
        public function create_payment($data = "", $header = '')
        {
            $headers = array(
                'Content-Type: application/json',
                'API-KEY: ' . $header['api'],
            );
            $url = $header['url'];
            $curl = curl_init();
            $data = json_encode($data);

            curl_setopt_array($curl, array(
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => $data,
                CURLOPT_HTTPHEADER => $headers,
            ));
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);

            $response = curl_exec($curl);
            curl_close($curl);
            return $response;
        }

        /** Update Order Status */
        public function update_order_status($order)
        {
            $transactionId = isset($_REQUEST['transactionId']) ? sanitize_text_field($_REQUEST['transactionId']) : '';
            if (empty($transactionId)) {
                $order->update_status('on-hold', __('No transaction ID received from Bongo Pay BD.', 'bongopay-bd'));
                return false;
            }

            $mode = $this->get_option('mode');
            $base_url = ($mode === 'test')
                ? 'https://sandbox.bongopaybd.com/'
                : 'https://secure.bongopaybd.com/';

            $data = array("transaction_id" => $transactionId);
            $header = array(
                "api" => sanitize_text_field($this->get_option('apikeys')),
                "url" => $base_url . "api/payment/verify"
            );

            $response = $this->create_payment($data, $header);
            $data = json_decode($response, true);

            if ($order->get_status() != 'completed') {
                if (isset($data['status']) && $data['status'] == "COMPLETED") {
                    $transaction_id = sanitize_text_field($data['transaction_id']);
                    $amount = sanitize_text_field($data['amount']);
                    $sender_number = sanitize_text_field($data['cus_email']);

                    if ($this->get_option('is_digital') === 'yes') {
                        $order->update_status('completed', __("Payment Completed: {$transaction_id}", 'bongopay-bd'));
                    } else {
                        $order->update_status('processing', __("Payment Processing: {$transaction_id}", 'bongopay-bd'));
                    }

                    $order->reduce_order_stock();
                    $order->payment_complete();
                    $order->add_order_note("Bongo Pay BD Payment Confirmed (Txn ID: {$transaction_id}, Amount: {$amount})");
                    return true;
                } else {
                    $order->update_status('on-hold', __('Payment verification failed. Please check manually.', 'bongopay-bd'));
                }
            }
        }

        /** Webhook Handler */
        public function handle_webhook()
        {
            $order_id = isset($_GET['order_id']) ? sanitize_text_field($_GET['order_id']) : '';
            $order = wc_get_order($order_id);

            if ($order) {
                $this->update_order_status($order);
            }

            status_header(200);
            echo json_encode(['message' => 'Webhook received and processed.']);
            exit();
        }
    }

    /** Register Gateway */
    function bongopay_bd_add_gateway_class($gateways)
    {
        $gateways[] = 'WC_BongoPay_BD_Gateway';
        return $gateways;
    }
    add_filter('woocommerce_payment_gateways', 'bongopay_bd_add_gateway_class');
}
